<?php

declare(strict_types=1);

namespace Strix\BlueMedia\Helper;

use Shopware\Core\Checkout\Payment\DataAbstractionLayer\PaymentMethodRepositoryDecorator;
use Shopware\Core\Framework\Context;
use Shopware\Core\Framework\DataAbstractionLayer\EntityRepositoryInterface;
use Shopware\Core\Framework\DataAbstractionLayer\Exception\InconsistentCriteriaIdsException;
use Shopware\Core\Framework\Plugin\Util\PluginIdProvider;
use Shopware\Core\Framework\Uuid\Uuid;
use Strix\BlueMedia\BluePayment;
use Strix\BlueMedia\Entity\GatewayEntity;
use Strix\BlueMedia\Payment\BmSeparatedPayment;
use Strix\BlueMedia\Provider\GatewayProvider;

class BmSeparatedPaymentHelper
{
    /**
     * @var GatewayProvider
     */
    private $gatewayProvider;

    /**
     * @var EntityRepositoryInterface
     */
    private $paymentMethodRepository;

    /**
     * @var PluginIdProvider
     */
    private $pluginIdProvider;

    /**
     * @var EntityRepositoryInterface
     */
    private $bmGatewayRepository;

    /**
     * @param GatewayProvider $gatewayProvider
     * @param EntityRepositoryInterface $paymentMethodRepository
     * @param EntityRepositoryInterface $bmGatewayRepository
     * @param PluginIdProvider $pluginIdProvider
     */
    public function __construct(
        GatewayProvider $gatewayProvider,
        EntityRepositoryInterface $paymentMethodRepository,
        EntityRepositoryInterface $bmGatewayRepository,
        PluginIdProvider $pluginIdProvider
    ) {
        $this->gatewayProvider = $gatewayProvider;
        $this->paymentMethodRepository = $paymentMethodRepository;
        $this->bmGatewayRepository = $bmGatewayRepository;
        $this->pluginIdProvider = $pluginIdProvider;
    }

    /**
     * @param string $gatewayUuid
     * @param Context $context
     */
    public function synchronizeBmPaymentByGatewayUuid(string $gatewayUuid, Context $context): void
    {
        $gateway = null;
        try {
            $gateway = $this->gatewayProvider->getGatewayByGatewayUuid($gatewayUuid, [], $context);
        } catch (InconsistentCriteriaIdsException $e) {
        }

        if(empty($gateway)) {
            return;
        }

        if($gateway->getIsSeparatedMethod()) {
            if($gateway->getPaymentMethodId()) {
                $this->updateBmPaymentByGateway($gateway, $context);
            } else {
                $this->insertBmPaymentByGateway($gateway, $context);
            }
        } else {
            $this->removeBmPaymentByGateway($gateway, $context);
        }
    }

    /**
     * @param GatewayEntity $gateway
     * @param Context $context
     */
    private function updateBmPaymentByGateway(GatewayEntity $gateway, Context $context): void
    {
        $data = [
            'id' => $gateway->getPaymentMethodId(),
            'handlerIdentifier' => BmSeparatedPayment::class,
            'active' => !$gateway->getForceDisable() && $gateway->getGatewayStatus(),
            'pluginId' => $this->pluginIdProvider->getPluginIdByBaseClass(BluePayment::class, $context),
            'mediaId' => $gateway->getMediaId(),
            'name' => $gateway->getGatewayName(),
            'description' => $gateway->getGatewayDescription(),
        ];
        $this->paymentMethodRepository->upsert([$data], Context::createDefaultContext());
    }

    /**
     * @param GatewayEntity $gateway
     * @param Context $context
     */
    private function insertBmPaymentByGateway(GatewayEntity $gateway, Context $context): void
    {
        $paymentId = Uuid::randomHex();
        $updateGateway = [
            'id' => $gateway->getId(),
            'paymentMethod' => [
                'id' => $paymentId,
                'handlerIdentifier' => BmSeparatedPayment::class,
                'active' => !$gateway->getForceDisable() && $gateway->getGatewayStatus(),
                'pluginId' => $this->pluginIdProvider->getPluginIdByBaseClass(BluePayment::class, $context),
                'mediaId' => $gateway->getMediaId(),
                'name' => $gateway->getGatewayName(),
                'description' => $gateway->getGatewayDescription(),
            ],
        ];
        $this->bmGatewayRepository->upsert([$updateGateway], $context);
    }

    /**
     * @param GatewayEntity $gateway
     * @param Context $context
     */
    private function removeBmPaymentByGateway(GatewayEntity $gateway, Context $context): void
    {
        $paymentMethodId = $gateway->getPaymentMethodId();
        if($paymentMethodId) {
            try {
                if($this->paymentMethodRepository instanceof PaymentMethodRepositoryDecorator) {
                    $this->paymentMethodRepository->internalDelete([['id' => $paymentMethodId]], $context);
                } else {
                    $this->paymentMethodRepository->delete([['id' => $paymentMethodId]], $context);
                }
            } catch (\Exception $e) {
                $this->paymentMethodRepository->update([[
                    'id' => $paymentMethodId,
                    'active' => false,
                ]], $context);
            }
        }
    }
}
