<?php

declare(strict_types=1);

namespace Strix\BlueMedia\Provider;

use Shopware\Core\Framework\Context;
use Shopware\Core\Framework\DataAbstractionLayer\EntityCollection;
use Shopware\Core\Framework\DataAbstractionLayer\EntityRepositoryInterface;
use Shopware\Core\Framework\DataAbstractionLayer\Exception\InconsistentCriteriaIdsException;
use Shopware\Core\Framework\DataAbstractionLayer\Search\Criteria;
use Shopware\Core\Framework\DataAbstractionLayer\Search\EntitySearchResult;
use Shopware\Core\Framework\DataAbstractionLayer\Search\Filter\EqualsFilter;
use Shopware\Core\Framework\DataAbstractionLayer\Search\Sorting\FieldSorting;
use Shopware\Core\Framework\Uuid\Uuid;
use Strix\BlueMedia\Entity\GatewayEntity;

class GatewayProvider
{
    /**
     * @var EntityRepositoryInterface
     */
    private $gatewayRepository;

    /**
     * @param EntityRepositoryInterface $gatewayRepository
     */
    public function __construct(EntityRepositoryInterface $gatewayRepository)
    {
        $this->gatewayRepository = $gatewayRepository;
    }

    /**
     * @param int $gatewayId
     *
     * @return GatewayEntity|null
     */
    public function getGatewayByGatewayId(int $gatewayId): ?GatewayEntity
    {
        try {
            $criteria = new Criteria();
            $criteria->addFilter(new EqualsFilter('gatewayId', $gatewayId));
            $criteria->setLimit(1);
            $result = $this->gatewayRepository->search($criteria, Context::createDefaultContext());
            $elements = $result->getElements();

            if (empty($elements)) {
                return null;
            }

            return current($elements);
        } catch (InconsistentCriteriaIdsException $e){
            return null;
        }
    }

    /**
     * @return EntityCollection
     */
    public function getEnabledStandardGateways(): ?EntityCollection
    {
        try {
            $criteria = new Criteria();
            $criteria->addAssociation('media');
            $criteria->addFilter(new EqualsFilter('forceDisable', false));
            $criteria->addFilter(new EqualsFilter('gatewayStatus', true));
            $criteria->addFilter(new EqualsFilter('isSeparatedMethod', false));
            $criteria->addSorting(new FieldSorting('position', FieldSorting::ASCENDING));
            $result = $this->gatewayRepository->search($criteria, Context::createDefaultContext());
            return $result->getEntities();
        } catch (InconsistentCriteriaIdsException $e) {
            return null;
        }
    }

    /**
     * @param string $gatewayUuid
     * @param array $associations
     * @param Context|null $context
     * @return GatewayEntity|null
     * @throws InconsistentCriteriaIdsException
     */
    public function getGatewayByGatewayUuid(string $gatewayUuid, array $associations = [], ?Context $context = null): ?GatewayEntity
    {
        if(!$context) {
            $context = Context::createDefaultContext();
        }

        $criteria = new Criteria([$gatewayUuid]);
        $criteria->addAssociations($associations);
        /** @var EntitySearchResult $result */
        $result = $context->disableCache(
            function() use ($criteria, $context) {
                 return $this->gatewayRepository->search($criteria, $context);
            }
        );
        return $result->first();
    }

    /**
     * @param string $paymentUuid
     * @param array $associations
     * @return GatewayEntity|null
     * @throws InconsistentCriteriaIdsException
     */
    public function getGatewayByPaymentUuid(string $paymentUuid, array $associations = []): ?GatewayEntity
    {
        $criteria = new Criteria();
        $criteria->addAssociations($associations);
        $criteria->addFilter(new EqualsFilter('paymentMethodId', $paymentUuid));
        $result = $this->gatewayRepository->search($criteria, Context::createDefaultContext());
        return $result->first();
    }
}
