<?php

declare(strict_types=1);

namespace Strix\BlueMedia\Provider;

use Shopware\Core\Checkout\Payment\Cart\AsyncPaymentTransactionStruct;
use Shopware\Core\Framework\Context;
use Shopware\Core\Framework\DataAbstractionLayer\Search\Criteria;
use Shopware\Core\System\Currency\CurrencyEntity;
use Shopware\Core\Framework\DataAbstractionLayer\EntityRepositoryInterface;
use Shopware\Core\Framework\DataAbstractionLayer\Exception\InconsistentCriteriaIdsException;
use Shopware\Core\Framework\DataAbstractionLayer\Search\Filter\EqualsFilter;
use Shopware\Core\Checkout\Order\Aggregate\OrderTransaction\OrderTransactionEntity;

class TransactionDataProvider
{
    /**
     * @var EntityRepositoryInterface
     */
    private $currencyRepository;

    /**
     * @var EntityRepositoryInterface
     */
    private $orderTransactionRepository;

    /**
     * @param EntityRepositoryInterface $currencyRepository
     * @param EntityRepositoryInterface $orderTransactionRepository
     */
    public function __construct(
        EntityRepositoryInterface $currencyRepository,
        EntityRepositoryInterface $orderTransactionRepository
    ) {
        $this->currencyRepository = $currencyRepository;
        $this->orderTransactionRepository = $orderTransactionRepository;
    }

    /**
     * @param AsyncPaymentTransactionStruct $transactionStruct
     *
     * @return string
     */
    public function getOrderId(AsyncPaymentTransactionStruct $transactionStruct): string
    {
        return $transactionStruct->getOrder()->getId();
    }

    /**
     * @param AsyncPaymentTransactionStruct $transactionStruct
     *
     * @return string
     */
    public function getOrderAmount(AsyncPaymentTransactionStruct $transactionStruct): string
    {
        return number_format(round($transactionStruct->getOrder()->getAmountTotal(), 2), 2, '.', '');
    }

    /**
     * @param AsyncPaymentTransactionStruct $transactionStruct
     * @param Context $context
     *
     * @return string
     */
    public function getOrderCurrencyCode(AsyncPaymentTransactionStruct $transactionStruct, Context $context): string
    {
        $currencyId = $transactionStruct->getOrder()->getCurrencyId();

        try {
            /** @var CurrencyEntity $currency */
            $currency = $this->currencyRepository->search(new Criteria([$currencyId]), $context)->get($currencyId);
        } catch (InconsistentCriteriaIdsException $e) {
            return '';
        }
        return $currency->getIsoCode();
    }

    /**
     * @param AsyncPaymentTransactionStruct $transactionStruct
     *
     * @return string
     */
    public function getOrderCustomerEmail(AsyncPaymentTransactionStruct $transactionStruct): string
    {
        return $transactionStruct->getOrder()->getOrderCustomer()->getEmail();
    }

    /**
     * @param string $orderId
     * @param Context $context
     *
     * @return string|null
     */
    public function getTransactionIdByOrderId(string $orderId, Context $context): ?string
    {
        try {
            $criteria = new Criteria();
            $criteria->addFilter(new EqualsFilter('order_transaction.orderId', $orderId));
            /** @var OrderTransactionEntity|null $otransaction */
            $transaction = $this->orderTransactionRepository->search($criteria, $context)->first();
        } catch (InconsistentCriteriaIdsException $e){
            return null;
        }

        return $transaction ? $transaction->getId() : null;
    }

    /**
     * @param string $orderNumber
     * @param Context $context
     * @return OrderTransactionEntity|null
     */
    public function getTransactionByOrderNumber(string $orderNumber, Context $context): ?OrderTransactionEntity
    {
        try {
            $criteria = new Criteria();
            $criteria->addAssociation('order');
            $criteria->addFilter(new EqualsFilter('order.orderNumber', $orderNumber));
            return $this->orderTransactionRepository->search($criteria, $context)->first();
        } catch (InconsistentCriteriaIdsException $e){
            return null;
        }
    }
}
