<?php

declare(strict_types=1);

namespace Strix\BlueMedia\Storefront\Controller;

use Shopware\Core\Checkout\Payment\PaymentService;
use Shopware\Core\System\SalesChannel\SalesChannelContext;
use Strix\BlueMedia\Exception\InvalidRequestParamsException;
use Shopware\Storefront\Controller\StorefrontController;
use Strix\BlueMedia\Payment\BmStandardPayment;
use Strix\BlueMedia\Provider\TransactionDataProvider;
use Strix\BlueMedia\WebService\BmPaymentRequestBuilder;
use Symfony\Component\HttpFoundation\Request;
use Symfony\Component\HttpFoundation\Response;
use Strix\BlueMedia\Hydrator\OrderTransactionHydrator;
use Shopware\Core\Framework\DataAbstractionLayer\Exception\InconsistentCriteriaIdsException;
use Shopware\Core\Checkout\Payment\Exception\InvalidTransactionException;
use Strix\BlueMedia\Provider\ConfigProvider;
use Strix\BlueMedia\Validator\RequestHashValidator;
use Symfony\Component\Routing\Annotation\Route; //do not remove!
use Shopware\Core\Framework\Routing\Annotation\RouteScope; //do not remove!
use Strix\BlueMedia\Exception\InvalidRequestHashException;
use Symfony\Component\HttpFoundation\JsonResponse;
use Strix\BlueMedia\Processor\PaymentStatusProcessor;

/**
 * @RouteScope(scopes={"storefront"})
 */
class BluePaymentController extends StorefrontController
{
    /**
     * @var PaymentService
     */
    private $paymentService;

    /**
     * @var BmStandardPayment
     */
    private $paymentHandler;

    /**
     * @var OrderTransactionHydrator
     */
    private $orderTransactionHydrator;

    /**
     * @var ConfigProvider
     */
    private $configProvider;

    /**
     * @var RequestHashValidator
     */
    private $requestHashValidator;

    /**
     * @var PaymentStatusProcessor
     */
    private $paymentStatusProcessor;

    /**
     * @var TransactionDataProvider
     */
    private $transactionDataProvider;
    /**
     * @var BmPaymentRequestBuilder
     */
    private $bmPaymentRequestBuilder;

    /**
     * @param PaymentService $paymentService
     * @param BmStandardPayment $paymentHandler
     * @param OrderTransactionHydrator $orderTransactionHydrator
     * @param ConfigProvider $configProvider
     * @param RequestHashValidator $requestHashValidator
     * @param PaymentStatusProcessor $paymentStatusProcessor
     * @param TransactionDataProvider $transactionDataProvider
     * @param BmPaymentRequestBuilder $bmPaymentRequestBuilder
     */
    public function __construct(
        PaymentService $paymentService,
        BmStandardPayment $paymentHandler,
        OrderTransactionHydrator $orderTransactionHydrator,
        ConfigProvider $configProvider,
        RequestHashValidator $requestHashValidator,
        PaymentStatusProcessor $paymentStatusProcessor,
        TransactionDataProvider $transactionDataProvider,
        BmPaymentRequestBuilder $bmPaymentRequestBuilder
    ) {
        $this->paymentService = $paymentService;
        $this->paymentHandler = $paymentHandler;
        $this->orderTransactionHydrator = $orderTransactionHydrator;
        $this->configProvider = $configProvider;
        $this->requestHashValidator = $requestHashValidator;
        $this->paymentStatusProcessor = $paymentStatusProcessor;
        $this->transactionDataProvider = $transactionDataProvider;
        $this->bmPaymentRequestBuilder = $bmPaymentRequestBuilder;
    }

    /**
     * @Route("/blue-payment/finalize-transaction", defaults={"auth_required"=false}, name="bluepayment.finalize.transaction", methods={"GET"})
     *
     * @param Request $request
     * @param SalesChannelContext $salesChannelContext
     *
     * @return Response
     */
    public function finalizeTransaction(Request $request, SalesChannelContext $salesChannelContext): Response
    {
        try {
            $context = $salesChannelContext->getContext();
            $orderNumber = (string)$request->query->getAlnum('OrderID');
            $transaction = $this->transactionDataProvider->getTransactionByOrderNumber($orderNumber, $context);
            if(!$transaction) {
                throw new InvalidTransactionException($orderNumber);
            }
            $this->requestHashValidator->validateReturnHash($this->configProvider->getServiceId(), $orderNumber, (string)$request->query->getAlnum('Hash'));
            $paymentTransactionStruct = $this->orderTransactionHydrator->hydratePaymentTransactionStruct($transaction->getId(), $context);
            $this->paymentHandler->finalize($paymentTransactionStruct, $request, $salesChannelContext);
            $returnUrl = $this->configProvider->getCheckoutSuccessUrl() . $transaction->getOrderId();
            return $this->redirect($returnUrl);
        } catch (InvalidTransactionException | InconsistentCriteriaIdsException | InvalidRequestHashException $e) {
            return new JsonResponse(null, Response::HTTP_NO_CONTENT);
        }
    }

    /**
     * @Route("/blue-payment/status", defaults={"auth_required"=false, "csrf_protected"=false}, name="bluepayment.status", methods={"POST"})
     *
     * @param Request $request
     * @param SalesChannelContext $salesChannelContext
     *
     * @return Response
     */
    public function status(Request $request, SalesChannelContext $salesChannelContext): Response
    {
        try {
            $response = $this->paymentStatusProcessor->process($request, $salesChannelContext);
        } catch (InvalidRequestParamsException $e) {
            return new JsonResponse(null, Response::HTTP_NO_CONTENT);
        }
        return new Response($response);
    }
}
