<?php

declare(strict_types=1);

namespace Strix\BlueMedia\WebService;

class BmPaymentRequestBuilder
{
    /**
     * @var string
     */
    private $url;

    /**
     * @var
     */
    private $key;

    /**
     * @var array
     */
    private $fieldValues;

    /**
     * @var BmPaymentRequestFields
     */
    private $bmPaymentRequestFields;

    /**
     * @param BmPaymentRequestFields $bmPaymentRequestFields
     */
    public function __construct(
        BmPaymentRequestFields $bmPaymentRequestFields
    ) {
        $this->bmPaymentRequestFields = $bmPaymentRequestFields;
        $this->fieldValues = [];
    }

    /**
     * @param string $url
     * @return $this
     */
    public function setUrl(string $url): self
    {
        $this->url = $url;
        return $this;
    }

    /**
     * @param string $key
     * @return $this
     */
    public function setKey(string $key): self
    {
        $this->key = $key;
        return $this;
    }

    /**
     * @param int $serviceId
     * @return $this
     */
    public function setServiceId(int $serviceId): self
    {
        $this->fieldValues[BmPaymentRequestFields::SERVICE_ID_KEY] = $serviceId;
        return $this;
    }

    /**
     * @param string $orderId
     * @return $this
     */
    public function setOrderId(string $orderId): self
    {
        $this->fieldValues[BmPaymentRequestFields::ORDER_ID_KEY] = $orderId;
        return $this;
    }

    /**
     * @param string $amount
     * @return $this
     */
    public function setAmount(string $amount): self
    {
        $this->fieldValues[BmPaymentRequestFields::AMOUNT_KEY] = $amount;
        return $this;
    }

    /**
     * @param int $description
     * @return $this
     */
    public function setDescription(int $description): self
    {
        $this->fieldValues[BmPaymentRequestFields::DESCRIPTION_KEY] = $description;
        return $this;
    }

    /**
     * @param int $gatewayId
     * @return $this
     */
    public function setGatewayId(int $gatewayId): self
    {
        $this->fieldValues[BmPaymentRequestFields::GATEWAY_ID_KEY] = $gatewayId;
        return $this;
    }

    /**
     * @param string $currency
     * @return $this
     */
    public function setCurrency(string $currency): self
    {
        $this->fieldValues[BmPaymentRequestFields::CURRENCY_KEY] = $currency;
        return $this;
    }

    /**
     * @param string $customerEmail
     * @return $this
     */
    public function setCustomerEmail(string $customerEmail): self
    {
        $this->fieldValues[BmPaymentRequestFields::CUSTOMER_EMAIL_KEY] = $customerEmail;
        return $this;
    }

    /**
     * @param int $validityTime
     * @return $this
     */
    public function setValidityTime(int $validityTime): self
    {
        $this->fieldValues[BmPaymentRequestFields::VALIDITY_TIME_KEY] = $validityTime;
        return $this;
    }

    /**
     * @param int $linkValidityTime
     * @return $this
     */
    public function setLinkValidityTime(int $linkValidityTime): self
    {
        $this->fieldValues[BmPaymentRequestFields::LINK_VALIDITY_TIME_KEY] = $linkValidityTime;
        return $this;
    }

    /**
     * @param string $hash
     * @return $this
     */
    private function setHash(string $hash): self
    {
        $this->fieldValues[BmPaymentRequestFields::HASH_KEY] = $hash;
        return $this;
    }

    /**
     * @param int $hours
     * @return $this
     */
    public function setValidityTimeout(int $hours): self
    {
        date_default_timezone_set('Europe/Warsaw');
        $expirationTime = date('Y-m-d H:i:s', strtotime("+{$hours} hour"));
        $this->setValidityTime($expirationTime);
    }

    /**
     * @return string
     */
    private function generateHash(): string
    {
        $this->fieldValues = $this->bmPaymentRequestFields->sortFields($this->fieldValues);
        $sorted = array_values($this->fieldValues);
        $sorted[] = $this->key;
        $hashString = implode('|', $sorted);
        return hash('sha256', $hashString);
    }

    /**
     * @return array
     */
    private function getRequestData(): array
    {
        $hash = $this->generateHash();
        $this->setHash($hash);
        return $this->fieldValues;
    }

    /**
     * @return string
     */
    public function build(): string
    {
        return $this->url.'?'.http_build_query( $this->getRequestData() );
    }
}
