<?php

declare(strict_types=1);

namespace Strix\BlueMedia\Hydrator;

use Shopware\Core\Checkout\Order\Aggregate\OrderTransaction\OrderTransactionEntity;
use Shopware\Core\Checkout\Payment\Cart\AsyncPaymentTransactionStruct;
use Shopware\Core\Checkout\Payment\Exception\InvalidTransactionException;
use Shopware\Core\Framework\DataAbstractionLayer\Exception\InconsistentCriteriaIdsException;
use Shopware\Core\Framework\Context;
use Shopware\Core\Framework\DataAbstractionLayer\EntityRepositoryInterface;
use Shopware\Core\Framework\DataAbstractionLayer\Search\Criteria;

class OrderTransactionHydrator
{
    /**
     * @var EntityRepositoryInterface
     */
    private $orderTransactionRepository;

    /**
     * @param EntityRepositoryInterface $orderTransactionRepository
     */
    public function __construct(EntityRepositoryInterface $orderTransactionRepository)
    {
        $this->orderTransactionRepository = $orderTransactionRepository;
    }

    /**
     * @param string $orderTransactionId
     * @param Context $context
     *
     * @return AsyncPaymentTransactionStruct
     * @throws InvalidTransactionException
     * @throws InconsistentCriteriaIdsException
     */
    public function hydratePaymentTransactionStruct(
        string $orderTransactionId,
        Context $context
    ): AsyncPaymentTransactionStruct {
        $orderTransaction = $this->getTransactionById($orderTransactionId, $context);

        return new AsyncPaymentTransactionStruct($orderTransaction, $orderTransaction->getOrder(), '');
    }

    /**
     * @param string $orderTransactionId
     * @param Context $context
     *
     * @return OrderTransactionEntity
     * @throws InconsistentCriteriaIdsException
     * @throws InvalidTransactionException
     */
    public function getTransactionById(string $orderTransactionId, Context $context): OrderTransactionEntity
    {
        $criteria = new Criteria([$orderTransactionId]);
        $criteria->addAssociation('order');
        /** @var OrderTransactionEntity|null $orderTransaction */
        $orderTransaction = $this->orderTransactionRepository->search($criteria, $context)->first();

        if ($orderTransaction === null) {
            throw new InvalidTransactionException($orderTransactionId);
        }

        return $orderTransaction;
    }
}
