<?php

declare(strict_types=1);

namespace Strix\BlueMedia\Hydrator;

use Strix\BlueMedia\Exception\InvalidRequestParamsException;
use Strix\BlueMedia\Provider\ConfigProvider;
use Strix\BlueMedia\Provider\HashProvider;
use Strix\BlueMedia\Provider\TransactionDataProvider;
use Symfony\Component\HttpFoundation\Request;
use Psr\Log\LoggerInterface;
use SimpleXMLElement;
use DOMDocument;

class PaymentStatusRequestHydrator
{
    /**
     * @var TransactionDataProvider
     */
    private $transactionDataProvider;

    /**
     * @var ConfigProvider
     */
    private $configProvider;

    /**
     * @var HashProvider
     */
    private $hashProvider;

    /**
     * @var LoggerInterface
     */
    private $logger;

    /**
     * @param TransactionDataProvider $transactionDataProvider
     * @param ConfigProvider $configProvider
     * @param HashProvider $hashProvider
     * @param LoggerInterface $logger
     */
    public function __construct(
        TransactionDataProvider $transactionDataProvider,
        ConfigProvider $configProvider,
        HashProvider $hashProvider,
        LoggerInterface $logger
    ) {
        $this->transactionDataProvider = $transactionDataProvider;
        $this->configProvider = $configProvider;
        $this->hashProvider = $hashProvider;
        $this->logger = $logger;
    }

    /**
     * @param Request $request
     *
     * @return SimpleXMLElement
     * @throws InvalidRequestParamsException
     */
    public function hydrateXmlDataFromRequest(Request $request): SimpleXMLElement
    {
        $paramTransactions = str_replace(' ', '+',$request->get('transactions'));
        if(empty($paramTransactions)){
            $this->logger->warning('Invalid Blue Media transactions request - cannot process payments');
            throw new InvalidRequestParamsException();
        }
        $base64transactions = base64_decode($paramTransactions);
        $this->logger->info($base64transactions);
        return simplexml_load_string($base64transactions);
    }

    /**
     * @param SimpleXMLElement $requestXml
     *
     * @return string
     */
    public function hydrateResponse(SimpleXMLElement $requestXml): string
    {
        $serviceId = $this->configProvider->getServiceId();
        $hashData = [];
        array_push($hashData, (string)$serviceId);

        $dom = new DOMDocument('1.0', 'UTF-8');
        $confirmationList = $dom->createElement('confirmationList');
        $domServiceID = $dom->createElement('serviceID', (string)$serviceId);
        $confirmationList->appendChild($domServiceID);

        $transactionsConfirmations = $dom->createElement('transactionsConfirmations');
        $confirmationList->appendChild($transactionsConfirmations);

        foreach ($requestXml->transactions->transaction as $transaction) {
            array_push($hashData, (string)$transaction->orderID, (string)$transaction->confirmation);
            $domTransactionConfirmed = $dom->createElement('transactionConfirmed');
            $transactionsConfirmations->appendChild($domTransactionConfirmed);
            $domOrderID = $dom->createElement('orderID', (string)$transaction->orderID);
            $domTransactionConfirmed->appendChild($domOrderID);

            $domConfirmation = $dom->createElement('confirmation', (string)$transaction->confirmation);
            $domTransactionConfirmed->appendChild($domConfirmation);
        }

        array_push($hashData, $this->configProvider->getSharedKey());
        $hash = $this->hashProvider->getHash($hashData);
        $domHash = $dom->createElement('hash', $hash);
        $confirmationList->appendChild($domHash);

        $dom->appendChild($confirmationList);
        return $dom->saveXML();
    }
}
