<?php

declare(strict_types=1);

namespace Strix\BlueMedia\Payment;

use Exception;
use Shopware\Core\Checkout\Order\Aggregate\OrderTransaction\OrderTransactionStateHandler;
use Shopware\Core\Checkout\Payment\Cart\AsyncPaymentTransactionStruct;
use Shopware\Core\Checkout\Payment\Cart\PaymentHandler\AsynchronousPaymentHandlerInterface;
use Shopware\Core\Checkout\Payment\Exception\AsyncPaymentProcessException;
use Shopware\Core\Framework\Validation\DataBag\RequestDataBag;
use Shopware\Core\System\SalesChannel\SalesChannelContext;
use Symfony\Component\HttpFoundation\RedirectResponse;
use Symfony\Component\HttpFoundation\Request;
use Strix\BlueMedia\Hydrator\PaymentRequestHydrator;
use Shopware\Core\Framework\Context;

class BmStandardPayment implements AsynchronousPaymentHandlerInterface
{
    /**
     * @var OrderTransactionStateHandler
     */
    private $transactionStateHandler;

    /**
     * @var PaymentRequestHydrator
     */
    private $paymentRequestHydrator;

    /**
     * @param OrderTransactionStateHandler $transactionStateHandler
     * @param PaymentRequestHydrator $paymentRequestHydrator
     */
    public function __construct(
        OrderTransactionStateHandler $transactionStateHandler,
        PaymentRequestHydrator $paymentRequestHydrator
    ) {
        $this->transactionStateHandler = $transactionStateHandler;
        $this->paymentRequestHydrator = $paymentRequestHydrator;
    }

    /**
     * @param AsyncPaymentTransactionStruct $transaction
     * @param RequestDataBag $dataBag
     * @param SalesChannelContext $salesChannelContext
     *
     * @return RedirectResponse
     * @throws AsyncPaymentProcessException
     */
    public function pay(
        AsyncPaymentTransactionStruct $transaction,
        RequestDataBag $dataBag,
        SalesChannelContext $salesChannelContext
    ): RedirectResponse {
        try {
            $redirectUrl = $this->prepareRedirectUrl($transaction, $salesChannelContext);
        } catch (Exception $e) {
            throw new AsyncPaymentProcessException(
                $transaction->getOrderTransaction()->getId(),
                'An error occurred during the communication with external payment gateway' . PHP_EOL . $e->getMessage()
            );
        }

        return new RedirectResponse($redirectUrl);
    }

    /**
     * @param AsyncPaymentTransactionStruct $transaction
     * @param Request $request
     * @param SalesChannelContext $salesChannelContext
     */
    public function finalize(
        AsyncPaymentTransactionStruct $transaction,
        Request $request,
        SalesChannelContext $salesChannelContext
    ): void {
    }

    /**
     * @param AsyncPaymentTransactionStruct $transaction
     * @param Context $context
     *
     * @return string
     */
    private function prepareRedirectUrl(AsyncPaymentTransactionStruct $transaction, SalesChannelContext $context): string
    {
        return $this->paymentRequestHydrator->hydratePaymentUrl($transaction, $context);
    }
}
