<?php

declare(strict_types=1);

namespace Strix\BlueMedia\Processor;

use Shopware\Core\Framework\Context;
use Shopware\Core\Framework\DataAbstractionLayer\EntityRepositoryInterface;
use SimpleXMLElement;
use Strix\BlueMedia\Provider\HashProvider;
use Strix\BlueMedia\Provider\ConfigProvider;
use Strix\BlueMedia\WebService\Client;
use Strix\BlueMedia\Exception\InvalidApiCallResponseException;
use Strix\BlueMedia\Provider\GatewayProvider;

class GatewaySynchronizationProcessor
{
    private const MESSAGE_ID_STRING_LENGTH = 32;

    /**
     * @var HashProvider
     */
    private $hashProvider;

    /**
     * @var ConfigProvider
     */
    private $configProvider;

    /**
     * @var Client
     */
    private $client;

    /**
     * @var EntityRepositoryInterface
     */
    private $gatewayRepository;

    /**
     * @var GatewayProvider
     */
    private $gatewayProvider;

    /**
     * @param HashProvider $hashProvider
     * @param ConfigProvider $configProvider
     * @param \Strix\BlueMedia\WebService\Client $client
     * @param EntityRepositoryInterface $gatewayRepository
     * @param GatewayProvider $gatewayProvider
     */
    public function __construct(
        HashProvider $hashProvider,
        ConfigProvider $configProvider,
        Client $client,
        EntityRepositoryInterface $gatewayRepository,
        GatewayProvider $gatewayProvider
    ) {
        $this->hashProvider = $hashProvider;
        $this->configProvider = $configProvider;
        $this->client = $client;
        $this->gatewayRepository = $gatewayRepository;
        $this->gatewayProvider = $gatewayProvider;
    }

    /**
     * @throws InvalidApiCallResponseException
     */
    public function processGatewaySynchronization(): void
    {
        $requestData = $this->getRequestData();
        $requestUrl = $this->configProvider->getGatewaysRequestUrl();
        $gateways = $this->client->call($requestUrl, $requestData);
        $this->saveGateways($gateways->gateway);
    }

    /**
     * @param SimpleXMLElement $gateways
     */
    private function saveGateways(SimpleXMLElement $gateways): void
    {
        foreach ($gateways as $gateway){
            $gatewayId = (int) $gateway->gatewayID;
            $gatewayData = [
                'gatewayCurrency' => 'PLN',
                'gatewayId' => $gatewayId,
                'bankName' => (string) $gateway->bankName,
                'gatewayName' => (string) $gateway->gatewayName,
                'gatewayType' => (string) $gateway->gatewayType,
                'gatewayLogoUrl' => (string) $gateway->iconURL
            ];

            $gatewayEntity = $this->gatewayProvider->getGatewayByGatewayId($gatewayId);
            if($gatewayEntity){
                $gatewayData['id'] = $gatewayEntity->getId();
            }

            $this->gatewayRepository->upsert([$gatewayData], Context::createDefaultContext());
        }
    }

    /**
     * @return array
     */
    private function getRequestData(): array
    {
        $serviceId = $this->configProvider->getServiceId();
        $messageId = $this->randomString(self::MESSAGE_ID_STRING_LENGTH);
        $sharedKey = $this->configProvider->getSharedKey();
        $hash = $this->hashProvider->getHash(
            [
                $serviceId,
                $messageId,
                $sharedKey
            ]
        );

        return [
            'ServiceID' => $serviceId,
            'MessageID' => $messageId,
            'Hash'      => $hash,
        ];
    }

    /**
     * @param $length
     *
     * @return string
     */
    private function randomString(int $length): string
    {
        $characters = '0123456789abcdefghijklmnopqrstuvwxyzABCDEFGHIJKLMNOPQRSTUVWXYZ';
        $randomString = '';
        for ($i = 0; $i < $length; $i++) {
            $randomString .= $characters[rand(0, strlen($characters) - 1)];
        }

        return $randomString;
    }
}
