<?php

declare(strict_types=1);

namespace Strix\BlueMedia\Processor;

use Shopware\Core\Checkout\Order\Aggregate\OrderTransaction\OrderTransactionStates;
use Symfony\Component\HttpFoundation\Request;
use Shopware\Core\System\SalesChannel\SalesChannelContext;
use Shopware\Core\Framework\Context;
use Shopware\Core\Checkout\Order\Aggregate\OrderTransaction\OrderTransactionStateHandler;
use Shopware\Core\Framework\DataAbstractionLayer\Exception\InconsistentCriteriaIdsException;
use Shopware\Core\System\StateMachine\Exception\IllegalTransitionException;
use Shopware\Core\System\StateMachine\Exception\StateMachineInvalidEntityIdException;
use Shopware\Core\System\StateMachine\Exception\StateMachineInvalidStateFieldException;
use Shopware\Core\System\StateMachine\Exception\StateMachineNotFoundException;
use Strix\BlueMedia\Hydrator\PaymentStatusRequestHydrator;
use Strix\BlueMedia\Hydrator\OrderTransactionHydrator;
use Strix\BlueMedia\Exception\InvalidRequestParamsException;
use Strix\BlueMedia\Exception\InvalidRequestHashException;
use Strix\BlueMedia\Exception\InvalidTransactionException;
use Strix\BlueMedia\Validator\RequestHashValidator;
use Strix\BlueMedia\Validator\TransactionValidator;
use Strix\BlueMedia\Provider\TransactionDataProvider;
use Shopware\Core\Checkout\Payment\Exception\InvalidTransactionException as ShopwareInvalidTransactionException;
use SimpleXMLElement;

class PaymentStatusProcessor
{
    const PAYMENT_STATUS_SUCCESS = 'SUCCESS';
    const PAYMENT_STATUS_PENDING = 'PENDING';
    const PAYMENT_STATUS_FAILURE = 'FAILURE';

    /**
     * @var PaymentStatusRequestHydrator
     */
    private $paymentStatusRequestHydrator;

    /**
     * @var RequestHashValidator
     */
    private $requestHashValidator;

    /**
     * @var OrderTransactionStateHandler
     */
    private $transactionStateHandler;

    /**
     * @var TransactionDataProvider
     */
    private $transactionDataProvider;

    /**
     * @var TransactionValidator
     */
    private $transactionValidator;

    /**
     * @var OrderTransactionHydrator
     */
    private $orderTransactionHydrator;

    /**
     * @param PaymentStatusRequestHydrator $paymentStatusRequestHydrator
     * @param RequestHashValidator $requestHashValidator
     * @param OrderTransactionStateHandler $transactionStateHandler
     * @param TransactionDataProvider $transactionDataProvider
     * @param TransactionValidator $transactionValidator
     * @param OrderTransactionHydrator $orderTransactionHydrator
     */
    public function __construct(
        PaymentStatusRequestHydrator $paymentStatusRequestHydrator,
        RequestHashValidator $requestHashValidator,
        OrderTransactionStateHandler $transactionStateHandler,
        TransactionDataProvider $transactionDataProvider,
        TransactionValidator $transactionValidator,
        OrderTransactionHydrator $orderTransactionHydrator
    ) {
        $this->paymentStatusRequestHydrator = $paymentStatusRequestHydrator;
        $this->requestHashValidator = $requestHashValidator;
        $this->transactionStateHandler = $transactionStateHandler;
        $this->transactionDataProvider = $transactionDataProvider;
        $this->transactionValidator = $transactionValidator;
        $this->orderTransactionHydrator = $orderTransactionHydrator;
    }

    /**
     * @param Request $request
     * @param SalesChannelContext $salesChannelContext
     *
     * @return string
     * @throws InvalidRequestParamsException
     */
    public function process(Request $request, SalesChannelContext $salesChannelContext): string
    {
        $context = $salesChannelContext->getContext();
        $requestXml = $this->paymentStatusRequestHydrator->hydrateXmlDataFromRequest($request);

        try {
            $this->requestHashValidator->validatePaymentStatusRequest($requestXml);
            $this->transactionValidator->validateServiceId((int) $requestXml->serviceID);
            foreach ($requestXml->transactions->transaction as $transaction) {
                $this->processTransaction($transaction, $context);
            }
        } catch (InvalidRequestHashException | InvalidRequestParamsException $e) {
            foreach ($requestXml->transactions->transaction as $transaction) {
                $transaction->confirmation = 'NOTCONFIRMED';
            }
        }
        return $this->paymentStatusRequestHydrator->hydrateResponse($requestXml);
    }

    /**
     * @param SimpleXMLElement $transaction
     * @param Context $context
     */
    private function processTransaction(SimpleXMLElement $transaction, Context $context): void
    {
        $paymentStatus = (string) $transaction->paymentStatus;
        if ($paymentStatus === static::PAYMENT_STATUS_SUCCESS || $paymentStatus === static::PAYMENT_STATUS_FAILURE) {
            try {
                $transactionEntity = $this->transactionDataProvider->getTransactionByOrderNumber((string) $transaction->orderID, $context);
                $this->transactionValidator->validateTransactionData($transactionEntity, $transaction, $context);
                $this->changeTransactionState($transactionEntity->getId(), $paymentStatus, $context);
                $transaction->confirmation = 'CONFIRMED';
            } catch (
            InconsistentCriteriaIdsException |
            StateMachineInvalidEntityIdException |
            StateMachineInvalidStateFieldException |
            InvalidTransactionException |
            StateMachineNotFoundException $e) {
                $transaction->confirmation = 'NOTCONFIRMED';
            } catch (IllegalTransitionException $e) {
            }
        } elseif ($paymentStatus === static::PAYMENT_STATUS_PENDING) {
            $transaction->confirmation = 'CONFIRMED';
        }
    }

    /**
     * @param string $transactionId
     * @param string $paymentStatus
     * @param Context $context
     *
     * @throws IllegalTransitionException
     * @throws InconsistentCriteriaIdsException
     * @throws StateMachineInvalidEntityIdException
     * @throws StateMachineInvalidStateFieldException
     * @throws StateMachineNotFoundException
     */
    private function changeTransactionState(string $transactionId, string $paymentStatus, Context $context): void
    {
        try {
            $transaction = $this->orderTransactionHydrator->getTransactionById($transactionId, $context);
            $transactionState = $transaction->getStateMachineState()->getTechnicalName();
        } catch (ShopwareInvalidTransactionException $e) {
            return;
        }

        if (!in_array($transactionState, [
            OrderTransactionStates::STATE_CANCELLED,
            OrderTransactionStates::STATE_OPEN
        ])) {
            return;
        }

        if ($paymentStatus === static::PAYMENT_STATUS_SUCCESS) {
            if ($transactionState === OrderTransactionStates::STATE_CANCELLED) {
                $this->transactionStateHandler->reopen($transactionId, $context);
            }
            $this->transactionStateHandler->pay($transactionId, $context);
        } elseif ($paymentStatus === static::PAYMENT_STATUS_FAILURE && $transactionState === OrderTransactionStates::STATE_OPEN) {
            $this->transactionStateHandler->cancel($transactionId, $context);
        }
    }
}
