<?php declare(strict_types=1);

namespace Strix\BlueMedia\Subscriber;

use Shopware\Core\Checkout\Payment\PaymentMethodEntity;
use Shopware\Core\Framework\DataAbstractionLayer\EntityCollection;
use Shopware\Core\Framework\DataAbstractionLayer\EntityWriteResult;
use Shopware\Core\Framework\DataAbstractionLayer\Event\EntityWrittenEvent;
use Shopware\Core\Framework\DataAbstractionLayer\Exception\InconsistentCriteriaIdsException;
use Shopware\Core\System\SalesChannel\Event\SalesChannelContextSwitchEvent;
use Shopware\Core\System\SalesChannel\SalesChannelContext;
use Shopware\Storefront\Page\Checkout\Confirm\CheckoutConfirmPageLoadedEvent;
use Strix\BlueMedia\Helper\BmSeparatedPaymentHelper;
use Strix\BlueMedia\Helper\SessionSelectedGatewayHelper;
use Strix\BlueMedia\Provider\BmPaymentProvider;
use Strix\BlueMedia\Provider\GatewayProvider;
use Symfony\Component\EventDispatcher\EventSubscriberInterface;

class CheckoutSubscriber implements EventSubscriberInterface
{
    const BM_EXTENSION_GATEWAY_KEY = 'BlueMediaGateways';

    const GATEWAY_FIELDS_TO_SYNCHRONIZE = ['gatewayStatus', 'gatewayName', 'gatewayDescription', 'isSeparatedMethod', 'forceDisable', 'mediaId'];

    /**
     * @var GatewayProvider
     */
    private $gatewayProvider;

    /**
     * @var SessionSelectedGatewayHelper
     */
    private $SessionSelectedGatewayHelper;

    /**
     * @var BmPaymentProvider
     */
    private $bmPaymentProvider;

    /**
     * @var BmSeparatedPaymentHelper
     */
    private $bmSeparatedPaymentHelper;

    /**
     * @param GatewayProvider $gatewayProvider
     * @param SessionSelectedGatewayHelper $SessionSelectedGatewayHelper
     * @param BmPaymentProvider $bmPaymentProvider
     * @param BmSeparatedPaymentHelper $bmPaymentHelper
     */
    public function __construct(
        GatewayProvider $gatewayProvider,
        SessionSelectedGatewayHelper $SessionSelectedGatewayHelper,
        BmPaymentProvider $bmPaymentProvider,
        BmSeparatedPaymentHelper $bmPaymentHelper
    ) {
        $this->gatewayProvider = $gatewayProvider;
        $this->SessionSelectedGatewayHelper = $SessionSelectedGatewayHelper;
        $this->bmPaymentProvider = $bmPaymentProvider;
        $this->bmSeparatedPaymentHelper = $bmPaymentHelper;
    }

    /**
     * @return array
     */
    public static function getSubscribedEvents(): array
    {
        return [
            CheckoutConfirmPageLoadedEvent::class => ['onConfirmPageLoaded', 1],
            SalesChannelContextSwitchEvent::class => ['onSalesChannelContextSwitch', 1],
            'blue_media_gateway.written' => ['onEntityWrittenContainer', 1],
        ];
    }

    /**
     * @param CheckoutConfirmPageLoadedEvent $event
     */
    public function onConfirmPageLoaded(CheckoutConfirmPageLoadedEvent $event): void
    {
        foreach($event->getPage()->getPaymentMethods()->getElements() as $paymentMethod)
        {
            if($paymentMethod->getId() == $this->bmPaymentProvider->getStandardPaymentMethodId()) {
                $this->addBlueMediaGateways($paymentMethod, $event->getSalesChannelContext());
            }
        }
    }

    /**
     * @param PaymentMethodEntity $paymentMethod
     * @param SalesChannelContext $salesChannelContext
     */
    private function addBlueMediaGateways(PaymentMethodEntity $paymentMethod, SalesChannelContext $salesChannelContext)
    {
        $bluemediaGateways = $this->gatewayProvider->getEnabledStandardGateways();
        $selectedGatewayId = $this->SessionSelectedGatewayHelper->getSelectedGatewayUuidd($salesChannelContext);
        if($bluemediaGateways && $selectedGatewayId) {
            $this->addSelectedGateway($bluemediaGateways, $selectedGatewayId, $salesChannelContext);
        }
        $paymentMethod->addExtension(self::BM_EXTENSION_GATEWAY_KEY, $bluemediaGateways);
    }

    /**
     * @param EntityCollection $bluemediaGateways
     * @param string $selectedGatewayId
     * @param SalesChannelContext $salesChannelContext
     */
    private function addSelectedGateway(EntityCollection $bluemediaGateways, string $selectedGatewayId, SalesChannelContext $salesChannelContext)
    {
        $selectedGateway = $bluemediaGateways->filterByProperty('id', $selectedGatewayId)->first();
        if($selectedGateway) {
            $selectedGateway->setPaymentMethodId($this->bmPaymentProvider->getStandardPaymentMethodId());
            $salesChannelContext->addExtension('bluemediaPaymentMethod', $selectedGateway);
        }
    }

    /**
     * @param SalesChannelContextSwitchEvent $event
     */
    public function onSalesChannelContextSwitch(SalesChannelContextSwitchEvent $event): void
    {
        $parameters = $event->getRequestDataBag()->all();
        if(array_key_exists('bluemediaPaymentMethod', $parameters)) {
            $this->SessionSelectedGatewayHelper->selectGateway($parameters['bluemediaPaymentMethod'], $event->getSalesChannelContext());
        }
    }

    /**
     * @param EntityWrittenEvent $event
     */
    public function onEntityWrittenContainer(EntityWrittenEvent $event): void
    {
        /**
         * @var EntityWriteResult[] $entityWriteResult
         */
        $result = $event->getWriteResults();
        foreach($result as $entityWriteResult) {
            switch($entityWriteResult->getOperation())
            {
                case EntityWriteResult::OPERATION_UPDATE:
                    if($this->checkSynchronizeNeeded(array_keys($entityWriteResult->getPayload()))) {
                        $this->bmSeparatedPaymentHelper->synchronizeBmPaymentByGatewayUuid($entityWriteResult->getPrimaryKey(), $event->getContext());
                    }
                    break;
                case EntityWriteResult::OPERATION_INSERT:
                case EntityWriteResult::OPERATION_DELETE:
                    $this->bmSeparatedPaymentHelper->synchronizeBmPaymentByGatewayUuid($entityWriteResult->getPrimaryKey(), $event->getContext());
                    break;
            }
        }
    }

    /**
     * @param array $modifiedFields
     * @return bool
     */
    private function checkSynchronizeNeeded(array $modifiedFields): bool
    {
        $fieldsToUpdate = array_intersect(self::GATEWAY_FIELDS_TO_SYNCHRONIZE, $modifiedFields);
        return !empty($fieldsToUpdate);
    }

}
