<?php

declare(strict_types=1);

namespace Strix\BlueMedia\Validator;

use Strix\BlueMedia\Provider\ConfigProvider;
use Strix\BlueMedia\Exception\InvalidRequestHashException;
use SimpleXMLElement;
use Strix\BlueMedia\Provider\HashProvider;

class RequestHashValidator
{
    /**
     * @var ConfigProvider
     */
    private $configProvider;

    /**
     * @var HashProvider
     */
    private $hashProvider;

    /**
     * @param ConfigProvider $configProvider
     * @param HashProvider $hashProvider
     */
    public function __construct(
        ConfigProvider $configProvider,
        HashProvider $hashProvider
    ) {
        $this->configProvider = $configProvider;
        $this->hashProvider = $hashProvider;
    }

    /**
     * @param int $serviceId
     * @param string $orderId
     * @param string $hash
     *
     * @throws InvalidRequestHashException
     */
    public function validateReturnHash(int $serviceId, string $orderId, string $hash): void
    {
        $hashParams = [$serviceId, $orderId, $this->configProvider->getSharedKey()];
        $generatedHash = $this->hashProvider->getHash($hashParams);

        if ($hash !== $generatedHash) {
            throw new InvalidRequestHashException();
        }
    }

    /**
     * @param SimpleXMLElement $requestXml
     *
     * @throws InvalidRequestHashException
     */
    public function validatePaymentStatusRequest(SimpleXMLElement $requestXml): void
    {
        $hashParams = [];
        $hashParams[] = (string) $this->configProvider->getServiceId();

        $this->extractXmlFields($requestXml->transactions->transaction, $hashParams);

        $hashParams[] = $this->configProvider->getSharedKey();
        $requestHash = (string) $requestXml->hash;
        $generatedHash = $this->hashProvider->getHash($hashParams);

        if ($generatedHash !== $requestHash) {
            throw new InvalidRequestHashException();
        }
    }

    /**
     * @param SimpleXMLElement $element
     * @param array $fields
     */
    private function extractXmlFields(SimpleXMLElement $element, array &$fields): void
    {
        foreach ((array) $element as $field) {
            if ($field instanceof SimpleXMLElement) {
                $this->extractXmlFields($field, $fields);
            } else {
                $fields[] = $field;
            }
        }
    }
}
