<?php

declare(strict_types=1);

namespace Strix\BlueMedia\Validator;

use Shopware\Core\Checkout\Order\Aggregate\OrderTransaction\OrderTransactionEntity;
use Shopware\Core\Framework\DataAbstractionLayer\EntityRepositoryInterface;
use Shopware\Core\Framework\DataAbstractionLayer\Search\Criteria;
use Shopware\Core\Framework\DataAbstractionLayer\Exception\InconsistentCriteriaIdsException;
use Strix\BlueMedia\Exception\InvalidRequestParamsException;
use Strix\BlueMedia\Exception\InvalidTransactionException;
use Shopware\Core\Framework\Context;
use Strix\BlueMedia\Provider\ConfigProvider;
use SimpleXMLElement;

class TransactionValidator
{
    /**
     * @var EntityRepositoryInterface
     */
    private $orderRepository;

    /**
     * @var EntityRepositoryInterface
     */
    private $currencyRepository;

    /**
     * @var ConfigProvider
     */
    private $configProvider;

    /**
     * @param EntityRepositoryInterface $orderRepository
     * @param EntityRepositoryInterface $currencyRepository
     * @param ConfigProvider $configProvider
     */
    public function __construct(
        EntityRepositoryInterface $orderRepository,
        EntityRepositoryInterface $currencyRepository,
        ConfigProvider $configProvider
    ) {
        $this->orderRepository = $orderRepository;
        $this->currencyRepository = $currencyRepository;
        $this->configProvider = $configProvider;
    }

    /**
     * @param OrderTransactionEntity|null $transactionEntity
     * @param SimpleXMLElement $transaction
     * @param Context $context
     *
     * @throws InvalidTransactionException
     */
    public function validateTransactionData(?OrderTransactionEntity $transactionEntity, SimpleXMLElement $transaction, Context $context): void
    {
        if(!$transactionEntity) {
            throw new InvalidTransactionException();
        }

        $order = $transactionEntity->getOrder();
        if(!$order) {
            throw new InvalidTransactionException();
        }

        if (
            !$this->isValidAmount((float) $transaction->amount, (float) $order->getAmountTotal())
            || !$this->isValidCurrency((string) $order->getCurrencyId(), (string) $transaction->currency, $context)
        ) {
            throw new InvalidTransactionException();
        }
    }

    /**
     * @param int $serviceId
     *
     * @throws InvalidRequestParamsException
     */
    public function validateServiceId(int $serviceId): void
    {
        if($serviceId !== $this->configProvider->getServiceId()){
            throw new InvalidRequestParamsException();
        }
    }

    /**
     * @param float $requestAmount
     * @param float $orderAmount
     *
     * @return bool
     */
    private function isValidAmount(float $requestAmount, float $orderAmount): bool
    {
        if (round($requestAmount, 2) === round($orderAmount, 2)) {
            return true;
        }
        return false;
    }

    /**
     * @param string $currencyId
     * @param string $currencyCode
     * @param Context $context
     *
     * @return bool
     */
    private function isValidCurrency(string $currencyId, string $currencyCode, Context $context): bool
    {
        try {
            $currency = $this->currencyRepository->search(new Criteria([$currencyId]), $context)->get($currencyId);
        } catch (InconsistentCriteriaIdsException $e) {
            return false;
        }
        return $currency->getIsoCode() === $currencyCode;
    }
}
